#include <archive.h>
#include <archive_entry.h>
#include <kj/async.h>

#include "async-io.hh"
#include "file-descriptor.hh"
#include "lix/libutil/c-calls.hh"
#include "lix/libutil/charptr-cast.hh"
#include "lix/libutil/file-system.hh"
#include "lix/libutil/logging.hh"
#include "lix/libutil/serialise.hh"
#include "result.hh"
#include "lix/libutil/tarfile.hh"

namespace nix {

static int callback_open(struct archive *, void * self)
{
    return ARCHIVE_OK;
}

static ssize_t callback_read(struct archive * archive, void * _self, const void * * buffer)
{
    auto self = static_cast<TarArchive *>(_self);
    *buffer = self->buffer.data();

    try {
        return self->source->read(charptr_cast<char *>(self->buffer.data()), self->buffer.size());
    } catch (EndOfFile &) {
        return 0;
    } catch (std::exception & err) { // NOLINT(lix-foreign-exceptions)
        // NOLINTNEXTLINE(lix-unsafe-c-calls): what() is a c string
        archive_set_error(archive, EIO, "Source threw exception: %s", err.what());
        return -1;
    }
}

static int callback_close(struct archive *, void * self)
{
    return ARCHIVE_OK;
}

void TarArchive::check(int err, const std::string & reason)
{
    if (err == ARCHIVE_EOF)
        throw EndOfFile("reached end of archive");
    else if (err != ARCHIVE_OK)
        throw Error(reason, archive_error_string(this->archive));
}

TarArchive::TarArchive(Source & source, bool raw) : buffer(65536)
{
    this->archive = archive_read_new();
    this->source = &source;

    if (!raw) {
        archive_read_support_filter_all(archive);
        archive_read_support_format_all(archive);
    } else {
        archive_read_support_filter_all(archive);
        archive_read_support_format_raw(archive);
        archive_read_support_format_empty(archive);
    }
    archive_read_set_option(archive, nullptr, "mac-ext", nullptr);
    check(archive_read_open(archive, (void *)this, callback_open, callback_read, callback_close), "Failed to open archive (%s)");
}


TarArchive::TarArchive(const Path & path)
{
    this->archive = archive_read_new();

    archive_read_support_filter_all(archive);
    archive_read_support_format_all(archive);
    archive_read_set_option(archive, nullptr, "mac-ext", nullptr);
    check(
        archive_read_open_filename(archive, requireCString(path), 16384),
        "failed to open archive: %s"
    );
}

void TarArchive::close()
{
    check(archive_read_close(this->archive), "Failed to close archive (%s)");
}

TarArchive::~TarArchive()
{
    if (this->archive) archive_read_free(this->archive);
}

static void extract_archive(TarArchive & archive, const Path & destDir)
{
    requireCString(destDir);

    int flags = ARCHIVE_EXTRACT_TIME
        | ARCHIVE_EXTRACT_SECURE_SYMLINKS
        | ARCHIVE_EXTRACT_SECURE_NODOTDOT;

    for (;;) {
        struct archive_entry * entry;
        int r = archive_read_next_header(archive.archive, &entry);
        if (r == ARCHIVE_EOF) break;
        auto name = archive_entry_pathname(entry);
        if (!name)
            throw Error("cannot get archive member name: %s", archive_error_string(archive.archive));
        if (r == ARCHIVE_WARN)
            printTaggedWarning("%1%", Uncolored(archive_error_string(archive.archive)));
        else
            archive.check(r);

        // NOLINTNEXTLINE(lix-unsafe-c-calls): destDir is checked, name is a c string
        archive_entry_copy_pathname(entry,
            (destDir + "/" + name).c_str());

        // sources can and do contain dirs with no rx bits
        if (archive_entry_filetype(entry) == AE_IFDIR && (archive_entry_mode(entry) & 0500) != 0500)
            archive_entry_set_mode(entry, archive_entry_mode(entry) | 0500);

        // Patch hardlink path
        const char *original_hardlink = archive_entry_hardlink(entry);
        if (original_hardlink) {
            // NOLINTNEXTLINE(lix-unsafe-c-calls): destDir is checked, name is a c string
            archive_entry_copy_hardlink(entry,
                (destDir + "/" + original_hardlink).c_str());
        }

        archive.check(archive_read_extract(archive.archive, entry, flags));
    }

    archive.close();
}

kj::Promise<Result<void>> unpackTarfile(AsyncInputStream & source, const Path & destDir)
try {
    Pipe pipe;
    pipe.create();

    auto thr = std::async(
        std::launch::async,
        [&](AutoCloseFD fd) {
            FdSource source(fd.get());
            auto archive = TarArchive(source);

            createDirs(destDir);
            extract_archive(archive, destDir);
        },
        std::move(pipe.readSide)
    );

    AsyncFdIoStream sink{std::move(pipe.writeSide)};
    TRY_AWAIT(source.drainInto(sink));
    thr.get();
    co_return result::success();
} catch (...) {
    co_return result::current_exception();
}

void unpackTarfile(const Path & tarFile, const Path & destDir)
{
    auto archive = TarArchive(tarFile);

    createDirs(destDir);
    extract_archive(archive, destDir);
}

}
