# Nextcloud {#module-services-nextcloud}

[Nextcloud](https://nextcloud.com/) is an open-source,
self-hostable cloud platform. The server setup can be automated using
[services.nextcloud](#opt-services.nextcloud.enable). A
desktop client is packaged at `pkgs.nextcloud-client`.

The current default by NixOS is `nextcloud31` which is also the latest
major version available.

## Basic usage {#module-services-nextcloud-basic-usage}

Nextcloud is a PHP-based application which requires an HTTP server
([`services.nextcloud`](#opt-services.nextcloud.enable)
and optionally supports
[`services.nginx`](#opt-services.nginx.enable)).

For the database, you can set
[`services.nextcloud.config.dbtype`](#opt-services.nextcloud.config.dbtype) to
either `sqlite` (the default), `mysql`, or `pgsql`. The simplest is `sqlite`,
which will be automatically created and managed by the application. For the
last two, you can easily create a local database by setting
[`services.nextcloud.database.createLocally`](#opt-services.nextcloud.database.createLocally)
to `true`, Nextcloud will automatically be configured to connect to it through
socket.

A very basic configuration may look like this:
```nix
{ pkgs, ... }:
{
  services.nextcloud = {
    enable = true;
    hostName = "nextcloud.tld";
    database.createLocally = true;
    config = {
      dbtype = "pgsql";
      adminpassFile = "/path/to/admin-pass-file";
    };
  };

  networking.firewall.allowedTCPPorts = [
    80
    443
  ];
}
```

The `hostName` option is used internally to configure an HTTP
server using [`PHP-FPM`](https://php-fpm.org/)
and `nginx`. The `config` attribute set is
used by the imperative installer and all values are written to an additional file
to ensure that changes can be applied by changing the module's options.

In case the application serves multiple domains (those are checked with
[`$_SERVER['HTTP_HOST']`](https://www.php.net/manual/en/reserved.variables.server.php))
it's needed to add them to
[`services.nextcloud.settings.trusted_domains`](#opt-services.nextcloud.settings.trusted_domains).

Auto updates for Nextcloud apps can be enabled using
[`services.nextcloud.autoUpdateApps`](#opt-services.nextcloud.autoUpdateApps.enable).

## `nextcloud-occ` {#module-services-nextcloud-occ}

The management command [`occ`](https://docs.nextcloud.com/server/stable/admin_manual/occ_command.html) can be
invoked by using the `nextcloud-occ` wrapper that's globally available on a system with Nextcloud enabled.

It requires elevated permissions to become the `nextcloud` user. Given the way the privilege
escalation is implemented, parameters passed via the environment to Nextcloud (e.g. `OC_PASS`) are
currently ignored.

Custom service units that need to run `nextcloud-occ` either need elevated privileges
or the systemd configuration from `nextcloud-setup.service` (recommended):

```nix
{ config, ... }:
{
  systemd.services.my-custom-service = {
    script = ''
      nextcloud-occ …
    '';
    serviceConfig = {
      inherit (config.systemd.services.nextcloud-cron.serviceConfig)
        User
        LoadCredential
        KillMode
        ;
    };
  };
}
```

Please note that the options required are subject to change. Please make sure to read the
release notes when upgrading.

## Common problems {#module-services-nextcloud-pitfalls-during-upgrade}

  - **General notes.**
    Unfortunately Nextcloud appears to be very stateful when it comes to
    managing its own configuration. The config file lives in the home directory
    of the `nextcloud` user (by default
    `/var/lib/nextcloud/config/config.php`) and is also used to
    track several states of the application (e.g., whether installed or not).

     All configuration parameters are also stored in
    {file}`/var/lib/nextcloud/config/override.config.php` which is generated by
    the module and linked from the store to ensure that all values from
    {file}`config.php` can be modified by the module.
    However {file}`config.php` manages the application's state and shouldn't be
    touched manually because of that.

    ::: {.warning}
    Don't delete {file}`config.php`! This file
    tracks the application's state and a deletion can cause unwanted
    side-effects!
    :::

    ::: {.warning}
    Don't rerun `nextcloud-occ maintenance:install`!
    This command tries to install the application
    and can cause unwanted side-effects!
    :::
  - **Multiple version upgrades.**
    Nextcloud doesn't allow to move more than one major-version forward. E.g., if you're on
    `v16`, you cannot upgrade to `v18`, you need to upgrade to
    `v17` first. This is ensured automatically as long as the
    [stateVersion](#opt-system.stateVersion) is declared properly. In that case
    the oldest version available (one major behind the one from the previous NixOS
    release) will be selected by default and the module will generate a warning that reminds
    the user to upgrade to latest Nextcloud *after* that deploy.
  - **`Error: Command "upgrade" is not defined.`**
    This error usually occurs if the initial installation
    ({command}`nextcloud-occ maintenance:install`) has failed. After that, the application
    is not installed, but the upgrade is attempted to be executed. Further context can
    be found in [NixOS/nixpkgs#111175](https://github.com/NixOS/nixpkgs/issues/111175).

    First of all, it makes sense to find out what went wrong by looking at the logs
    of the installation via {command}`journalctl -u nextcloud-setup` and try to fix
    the underlying issue.

    - If this occurs on an *existing* setup, this is most likely because
      the maintenance mode is active. It can be deactivated by running
      {command}`nextcloud-occ maintenance:mode --off`. It's advisable though to
      check the logs first on why the maintenance mode was activated.
    - ::: {.warning}
      Only perform the following measures on
      *freshly installed instances!*
      :::

      A re-run of the installer can be forced by *deleting*
      {file}`/var/lib/nextcloud/config/config.php`. This is the only time
      advisable because the fresh install doesn't have any state that can be lost.
      In case that doesn't help, an entire re-creation can be forced via
      {command}`rm -rf ~nextcloud/`.

  - **Server-side encryption.**
    Nextcloud supports [server-side encryption (SSE)](https://docs.nextcloud.com/server/latest/admin_manual/configuration_files/encryption_configuration.html).
    This is not an end-to-end encryption, but can be used to encrypt files that will be persisted
    to external storage such as S3.

  - **Issues with file permissions / unsafe path transitions**

    {manpage}`systemd-tmpfiles(8)` makes sure that the paths for

    * configuration (including declarative config)
    * data
    * app store
    * home directory itself (usually `/var/lib/nextcloud`)

    are properly set up. However, `systemd-tmpfiles` will refuse to do so
    if it detects an unsafe path transition, i.e. creating files/directories
    within a directory that is neither owned by `root` nor by `nextcloud`, the
    owning user of the files/directories to be created.

    Symptoms of that include

    * `config/override.config.php` not being updated (and the config file
      eventually being garbage-collected).
    * failure to read from application data.

    To work around that, please make sure that all directories in question
    are owned by `nextcloud:nextcloud`.

  - **`Failed to open stream: No such file or directory` after deploys**

    Symptoms are errors like this after a deployment that disappear after
    a few minutes:

    ```
    Warning: file_get_contents(/run/secrets/nextcloud_db_password): Failed to open stream: No such file or directory in /nix/store/lqw657xbh6h67ccv9cgv104qhcs1i2vw-nextcloud-config.php on line 11

    Warning: http_response_code(): Cannot set response code - headers already sent (output started at /nix/store/lqw657xbh6h67ccv9cgv104qhcs1i2vw-nextcloud-config.php:11) in /nix/store/ikxpaq7kjdhpr4w7cgl1n28kc2gvlhg6-nextcloud-29.0.7/lib/base.php on line 639
    Cannot decode /run/secrets/nextcloud_secrets, because: Syntax error
    ```

    This can happen if [](#opt-services.nextcloud.secretFile) or
    [](#opt-services.nextcloud.config.dbpassFile) are managed by
    [sops-nix](https://github.com/Mic92/sops-nix/).

    Here, `/run/secrets/nextcloud_secrets` is a symlink to
    `/run/secrets.d/N/nextcloud_secrets`. The `N` will be incremented
    when the sops-nix activation script runs, i.e.
    `/run/secrets.d/N` doesn't exist anymore after a deploy,
    only `/run/secrets.d/N+1`.

    PHP maintains a [cache for `realpath`](https://www.php.net/manual/en/ini.core.php#ini.realpath-cache-size)
    that still resolves to the old path which is causing
    the `No such file or directory` error. Interestingly,
    the cache isn't used for `file_exists` which is why this warning
    comes instead of the error from `nix_read_secret` in
    `override.config.php`.

    One option to work around this is to turn off the cache by setting
    the cache size to zero:

    ```nix
    { services.nextcloud.phpOptions."realpath_cache_size" = "0"; }
    ```

## Using an alternative webserver as reverse-proxy (e.g. `httpd`) {#module-services-nextcloud-httpd}

By default, `nginx` is used as reverse-proxy for `nextcloud`.
However, it's possible to use e.g. `httpd` by explicitly disabling
`nginx` using [](#opt-services.nginx.enable) and fixing the
settings `listen.owner` &amp; `listen.group` in the
[corresponding `phpfpm` pool](#opt-services.phpfpm.pools).

An exemplary configuration may look like this:
```nix
{
  config,
  lib,
  pkgs,
  ...
}:
{
  services.nginx.enable = false;
  services.nextcloud = {
    enable = true;
    hostName = "localhost";

    # further, required options
  };
  services.phpfpm.pools.nextcloud.settings = {
    "listen.owner" = config.services.httpd.user;
    "listen.group" = config.services.httpd.group;
  };
  services.httpd = {
    enable = true;
    adminAddr = "webmaster@localhost";
    extraModules = [ "proxy_fcgi" ];
    virtualHosts."localhost" = {
      documentRoot = config.services.nextcloud.package;
      extraConfig = ''
        <Directory "${config.services.nextcloud.package}">
          <FilesMatch "\.php$">
            <If "-f %{REQUEST_FILENAME}">
              SetHandler "proxy:unix:${config.services.phpfpm.pools.nextcloud.socket}|fcgi://localhost/"
            </If>
          </FilesMatch>
          <IfModule mod_rewrite.c>
            RewriteEngine On
            RewriteBase /
            RewriteRule ^index\.php$ - [L]
            RewriteCond %{REQUEST_FILENAME} !-f
            RewriteCond %{REQUEST_FILENAME} !-d
            RewriteRule . /index.php [L]
          </IfModule>
          DirectoryIndex index.php
          Require all granted
          Options +FollowSymLinks
        </Directory>
      '';
    };
  };
}
```

## Installing Apps and PHP extensions {#installing-apps-php-extensions-nextcloud}

Nextcloud apps are installed statefully through the web interface.
Some apps may require extra PHP extensions to be installed.
This can be configured with the [](#opt-services.nextcloud.phpExtraExtensions) setting.

Alternatively, extra apps can also be declared with the [](#opt-services.nextcloud.extraApps) setting.
When using this setting, apps can no longer be managed statefully because this can lead to Nextcloud updating apps
that are managed by Nix:

```nix
{ config, pkgs, ... }:
{
  services.nextcloud.extraApps = with config.services.nextcloud.package.packages.apps; {
    inherit user_oidc calendar contacts;
  };
}
```

Keep in mind that this is essentially a mirror of the apps from the appstore, but managed in
nixpkgs. This is by no means a curated list of apps that receive special testing on each update.

If you want automatic updates it is recommended that you use web interface to install apps.

## Known warnings {#module-services-nextcloud-known-warnings}

### Logreader application only supports "file" log_type {#module-services-nextcloud-warning-logreader}

This is because

* our module writes logs into the journal (`journalctl -t Nextcloud`)
* the Logreader application that allows reading logs in the admin panel is enabled
  by default and requires logs written to a file.

If you want to view logs in the admin panel,
set [](#opt-services.nextcloud.settings.log_type) to "file".

If you prefer logs in the journal, disable the logreader application to shut up the
"info". We can't really do that by default since whether apps are enabled/disabled
is part of the application's state and tracked inside the database.

## Maintainer information {#module-services-nextcloud-maintainer-info}

As stated in the previous paragraph, we must provide a clean upgrade-path for Nextcloud
since it cannot move more than one major version forward on a single upgrade. This chapter
adds some notes how Nextcloud updates should be rolled out in the future.

While minor and patch-level updates are no problem and can be done directly in the
package-expression (and should be backported to supported stable branches after that),
major-releases should be added in a new attribute (e.g. Nextcloud `v19.0.0`
should be available in `nixpkgs` as `pkgs.nextcloud19`).
To provide simple upgrade paths it's generally useful to backport those as well to stable
branches. As long as the package-default isn't altered, this won't break existing setups.
After that, the versioning-warning in the `nextcloud`-module should be
updated to make sure that the
[package](#opt-services.nextcloud.package)-option selects the latest version
on fresh setups.

If major-releases will be abandoned by upstream, we should check first if those are needed
in NixOS for a safe upgrade-path before removing those. In that case we should keep those
packages, but mark them as insecure in an expression like this (in
`<nixpkgs/pkgs/servers/nextcloud/default.nix>`):
```nix
# ...
{
  nextcloud17 = generic {
    version = "17.0.x";
    sha256 = "0000000000000000000000000000000000000000000000000000";
    eol = true;
  };
}
```

Ideally we should make sure that it's possible to jump two NixOS versions forward:
i.e. the warnings and the logic in the module should guard a user to upgrade from a
Nextcloud on e.g. 19.09 to a Nextcloud on 20.09.
