{ stdenv
, fetchFromGitHub
, fetchpatch
, fetchurl
}:

# This file is responsible for fetching the sage source and adding necessary patches.
# It does not actually build anything, it just copies the patched sources to $out.
# This is done because multiple derivations rely on these sources and they should
# all get the same sources with the same patches applied.

stdenv.mkDerivation rec {
  version = "10.3";
  pname = "sage-src";

  src = fetchFromGitHub {
    owner = "sagemath";
    repo = "sage";
    rev = version;
    hash = "sha256-OHtMv8t0RrP6R8XIREU+C1vpazeQLWa75wx9Mv6BN1U=";
  };

  # contains essential files (e.g., setup.cfg) generated by the bootstrap script.
  # TODO: investigate https://github.com/sagemath/sage/pull/35950
  configure-src = fetchurl {
    # the hash below is the tagged commit's _parent_. it can also be found by looking for
    # the "configure" asset at https://github.com/sagemath/sage/releases/tag/${version}
    url = "mirror://sageupstream/configure/configure-ab1a517b64b02bf15bbcb8d7c2d4d643bd5eff9b.tar.gz";
    hash = "sha256-pe9AxTM+gFSR4/eVfUzay+4bwjoubbYeDPc+avKjlaw=";
  };

  # Patches needed because of particularities of nix or the way this is packaged.
  # The goal is to upstream all of them and get rid of this list.
  nixPatches = [
    # Parallelize docubuild using subprocesses, fixing an isolation issue. See
    # https://groups.google.com/forum/#!topic/sage-packaging/YGOm8tkADrE
    ./patches/sphinx-docbuild-subprocesses.patch

    # After updating smypow to (https://github.com/sagemath/sage/issues/3360)
    # we can now set the cache dir to be within the .sage directory. This is
    # not strictly necessary, but keeps us from littering in the user's HOME.
    ./patches/sympow-cache.patch
  ];

  # Since sage unfortunately does not release bugfix releases, packagers must
  # fix those bugs themselves. This is for critical bugfixes, where "critical"
  # == "causes (transient) doctest failures / somebody complained".
  bugfixPatches = [
    # Sage uses mixed integer programs (MIPs) to find edge disjoint
    # spanning trees. For some reason, aarch64 glpk takes much longer
    # than x86_64 glpk to solve such MIPs. Since the MIP formulation
    # has "numerous problems" and will be replaced by a polynomial
    # algorithm soon, disable this test for now.
    # https://github.com/sagemath/sage/issues/34575
    ./patches/disable-slow-glpk-test.patch

    # https://github.com/sagemath/sage/pull/37489, landed in 10.4.beta1
    (fetchpatch {
      name = "quaternionalgebra-random-failure.patch";
      url = "https://github.com/sagemath/sage/commit/1c3f991b9d3c5778e409e5414c6cfcd456113f19.diff";
      hash = "sha256-uCXchYx26DdxTjR1k2748KCEHPnekKS2fAM7SpyhNvM=";
    })

    # https://github.com/sagemath/sage/pull/37763, landed in 10.4.beta2
    (fetchpatch {
      name = "scipy-fault-tolerance.patch";
      url = "https://github.com/sagemath/sage/commit/551139c09f26a5da96b1187c3f0dd17b8d80ef84.diff";
      hash = "sha256-PR4przrZ3ieHaW2nSY7l7VhNfrUupu9yCIrXpeyoAgg=";
    })

    # first two commits from https://github.com/sagemath/sage/pull/38100
    (fetchpatch {
      name = "sphinx-7.3-update.patch";
      url = "https://github.com/sagemath/sage/compare/10.4.beta7...d2932e142991b442a3eee5dc4f8949e3f7e6fe5c.patch";
      hash = "sha256-g6w0ImE0d2ocKnbGMQikcz9eI9r3/gESlA1NrQfGimk=";
    })
  ];

  # Patches needed because of package updates. We could just pin the versions of
  # dependencies, but that would lead to rebuilds, confusion and the burdons of
  # maintaining multiple versions of dependencies. Instead we try to make sage
  # compatible with never dependency versions when possible. All these changes
  # should come from or be proposed to upstream. This list will probably never
  # be empty since dependencies update all the time.
  packageUpgradePatches = [
    # https://github.com/sagemath/sage/pull/37492
    (fetchpatch {
      name = "singular-4.3.2p14-upgrade.patch";
      url = "https://github.com/sagemath/sage/commit/a0c56816b051e97da44ac0a4e4d4f6915cf7fa0f.diff";
      sha256 = "sha256-WGMmPeBoj2LUC+2qxWuaJL89QUuGt6axGvxWkpM9LYg=";
    })
  ];

  patches = nixPatches ++ bugfixPatches ++ packageUpgradePatches;

  # do not create .orig backup files if patch applies with fuzz
  patchFlags = [ "--no-backup-if-mismatch" "-p1" ];

  postPatch = ''
    # Make sure sage can at least be imported without setting any environment
    # variables. It won't be close to feature complete though.
    sed -i \
      "s|var(\"SAGE_ROOT\".*|var(\"SAGE_ROOT\", \"$out\")|" \
      src/sage/env.py

    # sage --docbuild unsets JUPYTER_PATH, which breaks our docbuilding
    # https://trac.sagemath.org/ticket/33650#comment:32
    sed -i "/export JUPYTER_PATH/d" src/bin/sage
  '';

  buildPhase = "# do nothing";

  installPhase = ''
    cp -r . "$out"
    tar xkzf ${configure-src} -C "$out"
    rm "$out/configure"
  '';
}
