{ system,
  pkgs,

  # Projects the test configuration into a the desired value; usually
  # the test runner: `config: config.test`.
  callTest,

}:
# The return value of this function will be an attrset with arbitrary depth and
# the `anything` returned by callTest at its test leafs.
# The tests not supported by `system` will be replaced with `{}`, so that
# `passthru.tests` can contain links to those without breaking on architectures
# where said tests are unsupported.
# Example callTest that just extracts the derivation from the test:
#   callTest = t: t.test;

with pkgs.lib;

let
  discoverTests = val:
    if isAttrs val
    then
      if hasAttr "test" val then callTest val
      else mapAttrs (n: s: if n == "passthru" then s else discoverTests s) val
    else if isFunction val
      then
        # Tests based on make-test-python.nix will return the second lambda
        # in that file, which are then forwarded to the test definition
        # following the `import make-test-python.nix` expression
        # (if it is a function).
        discoverTests (val { inherit system pkgs; })
      else val;
  handleTest = path: args:
    discoverTests (import path ({ inherit system pkgs; } // args));
  handleTestOn = systems: path: args:
    if elem system systems then handleTest path args
    else {};

  nixosLib = import ../lib {
    # Experimental features need testing too, but there's no point in warning
    # about it, so we enable the feature flag.
    featureFlags.minimalModules = {};
  };
  evalMinimalConfig = module: nixosLib.evalModules { modules = [ module ]; };

  inherit
    (rec {
      doRunTest = arg: ((import ../lib/testing-python.nix { inherit system pkgs; }).evalTest {
        imports = [ arg readOnlyPkgs ];
      }).config.result;
      findTests = tree:
        if tree?recurseForDerivations && tree.recurseForDerivations
        then
          mapAttrs
            (k: findTests)
            (builtins.removeAttrs tree ["recurseForDerivations"])
        else callTest tree;

      runTest = arg: let r = doRunTest arg; in findTests r;
      runTestOn = systems: arg:
        if elem system systems then runTest arg
        else {};
    })
    runTest
    runTestOn
    ;

  # Using a single instance of nixpkgs makes test evaluation faster.
  # To make sure we don't accidentally depend on a modified pkgs, we make the
  # related options read-only. We need to test the right configuration.
  #
  # If your service depends on a nixpkgs setting, first try to avoid that, but
  # otherwise, you can remove the readOnlyPkgs import and test your service as
  # usual.
  readOnlyPkgs =
    # TODO: We currently accept this for nixosTests, so that the `pkgs` argument
    #       is consistent with `pkgs` in `pkgs.nixosTests`. Can we reinitialize
    #       it with `allowAliases = false`?
    # warnIf pkgs.config.allowAliases "nixosTests: pkgs includes aliases."
    {
      _file = "${__curPos.file} readOnlyPkgs";
      _class = "nixosTest";
      node.pkgs = pkgs.pkgsLinux;
    };

in {

  # Testing the test driver
  nixos-test-driver = {
    extra-python-packages = handleTest ./nixos-test-driver/extra-python-packages.nix {};
    lib-extend = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./nixos-test-driver/lib-extend.nix {};
    node-name = runTest ./nixos-test-driver/node-name.nix;
    busybox = runTest ./nixos-test-driver/busybox.nix;
    driver-timeout = pkgs.runCommand "ensure-timeout-induced-failure" {
      failed = pkgs.testers.testBuildFailure ((runTest ./nixos-test-driver/timeout.nix).config.rawTestDerivation);
    } ''
      grep -F "timeout reached; test terminating" $failed/testBuildFailure.log
      # The program will always be terminated by SIGTERM (143) if it waits for the deadline thread.
      [[ 143 = $(cat $failed/testBuildFailure.exit) ]]
      touch $out
    '';
  };

  # NixOS vm tests and non-vm unit tests

  _3proxy = runTest ./3proxy.nix;
  aaaaxy = runTest ./aaaaxy.nix;
  acme = runTest ./acme.nix;
  acme-dns = handleTest ./acme-dns.nix {};
  adguardhome = runTest ./adguardhome.nix;
  aesmd = runTestOn ["x86_64-linux"] ./aesmd.nix;
  agate = runTest ./web-servers/agate.nix;
  agda = handleTest ./agda.nix {};
  airsonic = handleTest ./airsonic.nix {};
  akkoma = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./akkoma.nix {};
  akkoma-confined = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./akkoma.nix { confined = true; };
  alice-lg = handleTest ./alice-lg.nix {};
  allTerminfo = handleTest ./all-terminfo.nix {};
  alps = handleTest ./alps.nix {};
  amazon-init-shell = handleTest ./amazon-init-shell.nix {};
  amazon-ssm-agent = handleTest ./amazon-ssm-agent.nix {};
  amd-sev = runTest ./amd-sev.nix;
  anbox = runTest ./anbox.nix;
  angie-api = handleTest ./angie-api.nix {};
  anki-sync-server = handleTest ./anki-sync-server.nix {};
  anuko-time-tracker = handleTest ./anuko-time-tracker.nix {};
  apcupsd = handleTest ./apcupsd.nix {};
  apfs = runTest ./apfs.nix;
  appliance-repart-image = runTest ./appliance-repart-image.nix;
  apparmor = handleTest ./apparmor.nix {};
  archi = handleTest ./archi.nix {};
  armagetronad = handleTest ./armagetronad.nix {};
  artalk = handleTest ./artalk.nix {};
  atd = handleTest ./atd.nix {};
  atop = handleTest ./atop.nix {};
  atuin = handleTest ./atuin.nix {};
  audiobookshelf = handleTest ./audiobookshelf.nix {};
  auth-mysql = handleTest ./auth-mysql.nix {};
  authelia = handleTest ./authelia.nix {};
  avahi = handleTest ./avahi.nix {};
  avahi-with-resolved = handleTest ./avahi.nix { networkd = true; };
  ayatana-indicators = handleTest ./ayatana-indicators.nix {};
  babeld = handleTest ./babeld.nix {};
  bazarr = handleTest ./bazarr.nix {};
  bcachefs = handleTestOn ["x86_64-linux" "aarch64-linux"] ./bcachefs.nix {};
  beanstalkd = handleTest ./beanstalkd.nix {};
  bees = handleTest ./bees.nix {};
  benchexec = handleTest ./benchexec.nix {};
  binary-cache = handleTest ./binary-cache.nix {};
  bind = handleTest ./bind.nix {};
  bird = handleTest ./bird.nix {};
  birdwatcher = handleTest ./birdwatcher.nix {};
  bitcoind = handleTest ./bitcoind.nix {};
  bittorrent = handleTest ./bittorrent.nix {};
  blockbook-frontend = handleTest ./blockbook-frontend.nix {};
  blocky = handleTest ./blocky.nix {};
  boot = handleTestOn ["x86_64-linux" "aarch64-linux"] ./boot.nix {};
  bootspec = handleTestOn ["x86_64-linux"] ./bootspec.nix {};
  boot-stage1 = handleTest ./boot-stage1.nix {};
  borgbackup = handleTest ./borgbackup.nix {};
  botamusique = handleTest ./botamusique.nix {};
  bpf = handleTestOn ["x86_64-linux" "aarch64-linux"] ./bpf.nix {};
  bpftune = handleTest ./bpftune.nix {};
  breitbandmessung = handleTest ./breitbandmessung.nix {};
  brscan5 = handleTest ./brscan5.nix {};
  btrbk = handleTest ./btrbk.nix {};
  btrbk-doas = handleTest ./btrbk-doas.nix {};
  btrbk-no-timer = handleTest ./btrbk-no-timer.nix {};
  btrbk-section-order = handleTest ./btrbk-section-order.nix {};
  budgie = handleTest ./budgie.nix {};
  buildbot = handleTest ./buildbot.nix {};
  buildkite-agents = handleTest ./buildkite-agents.nix {};
  c2fmzq = handleTest ./c2fmzq.nix {};
  caddy = handleTest ./caddy.nix {};
  cadvisor = handleTestOn ["x86_64-linux"] ./cadvisor.nix {};
  cage = handleTest ./cage.nix {};
  cagebreak = handleTest ./cagebreak.nix {};
  calibre-web = handleTest ./calibre-web.nix {};
  calibre-server = handleTest ./calibre-server.nix {};
  castopod = handleTest ./castopod.nix {};
  cassandra_3_0 = handleTest ./cassandra.nix { testPackage = pkgs.cassandra_3_0; };
  cassandra_3_11 = handleTest ./cassandra.nix { testPackage = pkgs.cassandra_3_11; };
  cassandra_4 = handleTest ./cassandra.nix { testPackage = pkgs.cassandra_4; };
  centrifugo = runTest ./centrifugo.nix;
  ceph-multi-node = handleTestOn [ "aarch64-linux" "x86_64-linux" ] ./ceph-multi-node.nix {};
  ceph-single-node = handleTestOn [ "aarch64-linux" "x86_64-linux" ] ./ceph-single-node.nix {};
  ceph-single-node-bluestore = handleTestOn [ "aarch64-linux" "x86_64-linux" ] ./ceph-single-node-bluestore.nix {};
  certmgr = handleTest ./certmgr.nix {};
  cfssl = handleTestOn ["aarch64-linux" "x86_64-linux"] ./cfssl.nix {};
  cgit = handleTest ./cgit.nix {};
  charliecloud = handleTest ./charliecloud.nix {};
  chromium = (handleTestOn ["aarch64-linux" "x86_64-linux"] ./chromium.nix {}).stable or {};
  chrony = handleTestOn ["aarch64-linux" "x86_64-linux"] ./chrony.nix {};
  chrony-ptp = handleTestOn ["aarch64-linux" "x86_64-linux"] ./chrony-ptp.nix {};
  cinnamon = handleTest ./cinnamon.nix {};
  cinnamon-wayland = handleTest ./cinnamon-wayland.nix {};
  cjdns = handleTest ./cjdns.nix {};
  clatd = handleTest ./clatd.nix {};
  clickhouse = handleTest ./clickhouse.nix {};
  cloud-init = handleTest ./cloud-init.nix {};
  cloud-init-hostname = handleTest ./cloud-init-hostname.nix {};
  cloudlog = handleTest ./cloudlog.nix {};
  cntr = handleTestOn ["aarch64-linux" "x86_64-linux"] ./cntr.nix {};
  cockpit = handleTest ./cockpit.nix {};
  cockroachdb = handleTestOn ["x86_64-linux"] ./cockroachdb.nix {};
  code-server = handleTest ./code-server.nix {};
  coder = handleTest ./coder.nix {};
  collectd = handleTest ./collectd.nix {};
  commafeed = handleTest ./commafeed.nix {};
  connman = handleTest ./connman.nix {};
  consul = handleTest ./consul.nix {};
  consul-template = handleTest ./consul-template.nix {};
  containers-bridge = handleTest ./containers-bridge.nix {};
  containers-custom-pkgs.nix = handleTest ./containers-custom-pkgs.nix {};
  containers-ephemeral = handleTest ./containers-ephemeral.nix {};
  containers-extra_veth = handleTest ./containers-extra_veth.nix {};
  containers-hosts = handleTest ./containers-hosts.nix {};
  containers-imperative = handleTest ./containers-imperative.nix {};
  containers-ip = handleTest ./containers-ip.nix {};
  containers-macvlans = handleTest ./containers-macvlans.nix {};
  containers-names = handleTest ./containers-names.nix {};
  containers-nested = handleTest ./containers-nested.nix {};
  containers-physical_interfaces = handleTest ./containers-physical_interfaces.nix {};
  containers-portforward = handleTest ./containers-portforward.nix {};
  containers-reloadable = handleTest ./containers-reloadable.nix {};
  containers-require-bind-mounts = handleTest ./containers-require-bind-mounts.nix {};
  containers-restart_networking = handleTest ./containers-restart_networking.nix {};
  containers-tmpfs = handleTest ./containers-tmpfs.nix {};
  containers-unified-hierarchy = handleTest ./containers-unified-hierarchy.nix {};
  convos = handleTest ./convos.nix {};
  corerad = handleTest ./corerad.nix {};
  coturn = handleTest ./coturn.nix {};
  couchdb = handleTest ./couchdb.nix {};
  crabfit = handleTest ./crabfit.nix {};
  cri-o = handleTestOn ["aarch64-linux" "x86_64-linux"] ./cri-o.nix {};
  cups-pdf = handleTest ./cups-pdf.nix {};
  curl-impersonate = handleTest ./curl-impersonate.nix {};
  custom-ca = handleTest ./custom-ca.nix {};
  croc = handleTest ./croc.nix {};
  darling = handleTest ./darling.nix {};
  dae = handleTest ./dae.nix {};
  davis = handleTest ./davis.nix {};
  db-rest = handleTest ./db-rest.nix {};
  dconf = handleTest ./dconf.nix {};
  deconz = handleTest ./deconz.nix {};
  deepin = handleTest ./deepin.nix {};
  deluge = handleTest ./deluge.nix {};
  dendrite = handleTest ./matrix/dendrite.nix {};
  devpi-server = handleTest ./devpi-server.nix {};
  dex-oidc = handleTest ./dex-oidc.nix {};
  dhparams = handleTest ./dhparams.nix {};
  disable-installer-tools = handleTest ./disable-installer-tools.nix {};
  discourse = handleTest ./discourse.nix {};
  dnscrypt-proxy2 = handleTestOn ["x86_64-linux"] ./dnscrypt-proxy2.nix {};
  dnscrypt-wrapper = runTestOn ["x86_64-linux"] ./dnscrypt-wrapper;
  dnsdist = import ./dnsdist.nix { inherit pkgs runTest; };
  doas = handleTest ./doas.nix {};
  docker = handleTestOn ["aarch64-linux" "x86_64-linux"] ./docker.nix {};
  docker-rootless = handleTestOn ["aarch64-linux" "x86_64-linux"] ./docker-rootless.nix {};
  docker-registry = handleTest ./docker-registry.nix {};
  docker-tools = handleTestOn ["x86_64-linux"] ./docker-tools.nix {};
  docker-tools-cross = handleTestOn ["x86_64-linux" "aarch64-linux"] ./docker-tools-cross.nix {};
  docker-tools-overlay = handleTestOn ["x86_64-linux"] ./docker-tools-overlay.nix {};
  documize = handleTest ./documize.nix {};
  documentation = pkgs.callPackage ../modules/misc/documentation/test.nix { inherit nixosLib; };
  doh-proxy-rust = handleTest ./doh-proxy-rust.nix {};
  dokuwiki = handleTest ./dokuwiki.nix {};
  dolibarr = handleTest ./dolibarr.nix {};
  domination = handleTest ./domination.nix {};
  dovecot = handleTest ./dovecot.nix {};
  drawterm = discoverTests (import ./drawterm.nix);
  drbd = handleTest ./drbd.nix {};
  dublin-traceroute = handleTest ./dublin-traceroute.nix {};
  earlyoom = handleTestOn ["x86_64-linux"] ./earlyoom.nix {};
  early-mount-options = handleTest ./early-mount-options.nix {};
  ec2-config = (handleTestOn ["x86_64-linux"] ./ec2.nix {}).boot-ec2-config or {};
  ec2-nixops = (handleTestOn ["x86_64-linux"] ./ec2.nix {}).boot-ec2-nixops or {};
  ecryptfs = handleTest ./ecryptfs.nix {};
  fscrypt = handleTest ./fscrypt.nix {};
  fastnetmon-advanced = runTest ./fastnetmon-advanced.nix;
  ejabberd = handleTest ./xmpp/ejabberd.nix {};
  elk = handleTestOn ["x86_64-linux"] ./elk.nix {};
  emacs-daemon = handleTest ./emacs-daemon.nix {};
  endlessh = handleTest ./endlessh.nix {};
  endlessh-go = handleTest ./endlessh-go.nix {};
  engelsystem = handleTest ./engelsystem.nix {};
  enlightenment = handleTest ./enlightenment.nix {};
  env = handleTest ./env.nix {};
  envfs = handleTest ./envfs.nix {};
  envoy = handleTest ./envoy.nix {};
  ergo = handleTest ./ergo.nix {};
  ergochat = handleTest ./ergochat.nix {};
  eris-server = handleTest ./eris-server.nix {};
  esphome = handleTest ./esphome.nix {};
  etc = pkgs.callPackage ../modules/system/etc/test.nix { inherit evalMinimalConfig; };
  activation = pkgs.callPackage ../modules/system/activation/test.nix { };
  activation-var = runTest ./activation/var.nix;
  activation-nix-channel = runTest ./activation/nix-channel.nix;
  activation-etc-overlay-mutable = runTest ./activation/etc-overlay-mutable.nix;
  activation-etc-overlay-immutable = runTest ./activation/etc-overlay-immutable.nix;
  activation-perlless = runTest ./activation/perlless.nix;
  etcd = handleTestOn [ "aarch64-linux" "x86_64-linux" ] ./etcd/etcd.nix {};
  etcd-cluster = handleTestOn [ "aarch64-linux" "x86_64-linux" ] ./etcd/etcd-cluster.nix {};
  etebase-server = handleTest ./etebase-server.nix {};
  etesync-dav = handleTest ./etesync-dav.nix {};
  evcc = handleTest ./evcc.nix {};
  fail2ban = handleTest ./fail2ban.nix { };
  fakeroute = handleTest ./fakeroute.nix {};
  fancontrol = handleTest ./fancontrol.nix {};
  fanout = handleTest ./fanout.nix {};
  fcitx5 = handleTest ./fcitx5 {};
  fenics = handleTest ./fenics.nix {};
  ferm = handleTest ./ferm.nix {};
  ferretdb = handleTest ./ferretdb.nix {};
  filesender = handleTest ./filesender.nix {};
  filesystems-overlayfs = runTest ./filesystems-overlayfs.nix;
  firefly-iii = handleTest ./firefly-iii.nix {};
  firefox = handleTest ./firefox.nix { firefoxPackage = pkgs.firefox; };
  firefox-beta = handleTest ./firefox.nix { firefoxPackage = pkgs.firefox-beta; };
  firefox-devedition = handleTest ./firefox.nix { firefoxPackage = pkgs.firefox-devedition; };
  firefox-esr    = handleTest ./firefox.nix { firefoxPackage = pkgs.firefox-esr; }; # used in `tested` job
  firefox-esr-115 = handleTest ./firefox.nix { firefoxPackage = pkgs.firefox-esr-115; };
  firefoxpwa = handleTest ./firefoxpwa.nix {};
  firejail = handleTest ./firejail.nix {};
  firewall = handleTest ./firewall.nix { nftables = false; };
  firewall-nftables = handleTest ./firewall.nix { nftables = true; };
  fish = handleTest ./fish.nix {};
  flannel = handleTestOn ["x86_64-linux"] ./flannel.nix {};
  floorp = handleTest ./firefox.nix { firefoxPackage = pkgs.floorp; };
  fluentd = handleTest ./fluentd.nix {};
  fluidd = handleTest ./fluidd.nix {};
  fontconfig-default-fonts = handleTest ./fontconfig-default-fonts.nix {};
  forgejo = handleTest ./forgejo.nix { };
  freenet = handleTest ./freenet.nix {};
  freeswitch = handleTest ./freeswitch.nix {};
  freetube = discoverTests (import ./freetube.nix);
  freshrss-sqlite = handleTest ./freshrss-sqlite.nix {};
  freshrss-pgsql = handleTest ./freshrss-pgsql.nix {};
  freshrss-http-auth = handleTest ./freshrss-http-auth.nix {};
  freshrss-none-auth = handleTest ./freshrss-none-auth.nix {};
  frigate = handleTest ./frigate.nix {};
  frp = handleTest ./frp.nix {};
  frr = handleTest ./frr.nix {};
  fsck = handleTest ./fsck.nix {};
  fsck-systemd-stage-1 = handleTest ./fsck.nix { systemdStage1 = true; };
  ft2-clone = handleTest ./ft2-clone.nix {};
  legit = handleTest ./legit.nix {};
  mimir = handleTest ./mimir.nix {};
  garage = handleTest ./garage {};
  gemstash = handleTest ./gemstash.nix {};
  geoserver = runTest ./geoserver.nix;
  gerrit = handleTest ./gerrit.nix {};
  geth = handleTest ./geth.nix {};
  ghostunnel = handleTest ./ghostunnel.nix {};
  gitdaemon = handleTest ./gitdaemon.nix {};
  gitea = handleTest ./gitea.nix { giteaPackage = pkgs.gitea; };
  github-runner = handleTest ./github-runner.nix {};
  gitlab = runTest ./gitlab.nix;
  gitolite = handleTest ./gitolite.nix {};
  gitolite-fcgiwrap = handleTest ./gitolite-fcgiwrap.nix {};
  glusterfs = handleTest ./glusterfs.nix {};
  gnome = handleTest ./gnome.nix {};
  gnome-extensions = handleTest ./gnome-extensions.nix {};
  gnome-flashback = handleTest ./gnome-flashback.nix {};
  gnome-xorg = handleTest ./gnome-xorg.nix {};
  gns3-server = handleTest ./gns3-server.nix {};
  gnupg = handleTest ./gnupg.nix {};
  go-neb = handleTest ./go-neb.nix {};
  gobgpd = handleTest ./gobgpd.nix {};
  gocd-agent = handleTest ./gocd-agent.nix {};
  gocd-server = handleTest ./gocd-server.nix {};
  gollum = handleTest ./gollum.nix {};
  gonic = handleTest ./gonic.nix {};
  google-oslogin = handleTest ./google-oslogin {};
  goss = handleTest ./goss.nix {};
  gotify-server = handleTest ./gotify-server.nix {};
  gotosocial = runTest ./web-apps/gotosocial.nix;
  grafana = handleTest ./grafana {};
  grafana-agent = handleTest ./grafana-agent.nix {};
  graphite = handleTest ./graphite.nix {};
  graylog = handleTest ./graylog.nix {};
  grocy = handleTest ./grocy.nix {};
  grow-partition = runTest ./grow-partition.nix;
  grub = handleTest ./grub.nix {};
  guacamole-server = handleTest ./guacamole-server.nix {};
  guix = handleTest ./guix {};
  gvisor = handleTest ./gvisor.nix {};
  hadoop = import ./hadoop { inherit handleTestOn; package=pkgs.hadoop; };
  hadoop_3_2 = import ./hadoop { inherit handleTestOn; package=pkgs.hadoop_3_2; };
  hadoop2 = import ./hadoop { inherit handleTestOn; package=pkgs.hadoop2; };
  haka = handleTest ./haka.nix {};
  haste-server = handleTest ./haste-server.nix {};
  haproxy = handleTest ./haproxy.nix {};
  hardened = handleTest ./hardened.nix {};
  harmonia = runTest ./harmonia.nix;
  headscale = handleTest ./headscale.nix {};
  healthchecks = handleTest ./web-apps/healthchecks.nix {};
  hbase2 = handleTest ./hbase.nix { package=pkgs.hbase2; };
  hbase_2_4 = handleTest ./hbase.nix { package=pkgs.hbase_2_4; };
  hbase3 = handleTest ./hbase.nix { package=pkgs.hbase3; };
  hddfancontrol = handleTest ./hddfancontrol.nix {};
  hedgedoc = handleTest ./hedgedoc.nix {};
  herbstluftwm = handleTest ./herbstluftwm.nix {};
  homepage-dashboard = handleTest ./homepage-dashboard.nix {};
  honk = runTest ./honk.nix;
  installed-tests = pkgs.recurseIntoAttrs (handleTest ./installed-tests {});
  invidious = handleTest ./invidious.nix {};
  isolate = handleTest ./isolate.nix {};
  livebook-service = handleTest ./livebook-service.nix {};
  pyload = handleTest ./pyload.nix {};
  oci-containers = handleTestOn ["aarch64-linux" "x86_64-linux"] ./oci-containers.nix {};
  odoo = handleTest ./odoo.nix {};
  odoo15 = handleTest ./odoo.nix { package = pkgs.odoo15; };
  # 9pnet_virtio used to mount /nix partition doesn't support
  # hibernation. This test happens to work on x86_64-linux but
  # not on other platforms.
  hibernate = handleTestOn ["x86_64-linux"] ./hibernate.nix {};
  hibernate-systemd-stage-1 = handleTestOn ["x86_64-linux"] ./hibernate.nix { systemdStage1 = true; };
  hitch = handleTest ./hitch {};
  hledger-web = handleTest ./hledger-web.nix {};
  hocker-fetchdocker = handleTest ./hocker-fetchdocker {};
  hockeypuck = handleTest ./hockeypuck.nix { };
  home-assistant = handleTest ./home-assistant.nix {};
  hostname = handleTest ./hostname.nix {};
  hound = handleTest ./hound.nix {};
  hub = handleTest ./git/hub.nix {};
  hydra = handleTest ./hydra {};
  i3wm = handleTest ./i3wm.nix {};
  icingaweb2 = handleTest ./icingaweb2.nix {};
  iftop = handleTest ./iftop.nix {};
  incron = handleTest ./incron.nix {};
  incus = pkgs.recurseIntoAttrs (handleTest ./incus { inherit handleTestOn; inherit (pkgs) incus; });
  incus-lts = pkgs.recurseIntoAttrs (handleTest ./incus { inherit handleTestOn; });
  influxdb = handleTest ./influxdb.nix {};
  influxdb2 = handleTest ./influxdb2.nix {};
  initrd-network-openvpn = handleTestOn [ "x86_64-linux" "i686-linux" ] ./initrd-network-openvpn {};
  initrd-network-ssh = handleTest ./initrd-network-ssh {};
  initrd-luks-empty-passphrase = handleTest ./initrd-luks-empty-passphrase.nix {};
  initrdNetwork = handleTest ./initrd-network.nix {};
  initrd-secrets = handleTest ./initrd-secrets.nix {};
  initrd-secrets-changing = handleTest ./initrd-secrets-changing.nix {};
  input-remapper = handleTest ./input-remapper.nix {};
  inspircd = handleTest ./inspircd.nix {};
  installer = handleTest ./installer.nix {};
  installer-systemd-stage-1 = handleTest ./installer-systemd-stage-1.nix {};
  intune = handleTest ./intune.nix {};
  invoiceplane = handleTest ./invoiceplane.nix {};
  iodine = handleTest ./iodine.nix {};
  ipv6 = handleTest ./ipv6.nix {};
  iscsi-multipath-root = handleTest ./iscsi-multipath-root.nix {};
  iscsi-root = handleTest ./iscsi-root.nix {};
  isso = handleTest ./isso.nix {};
  jackett = handleTest ./jackett.nix {};
  jellyfin = handleTest ./jellyfin.nix {};
  jenkins = handleTest ./jenkins.nix {};
  jenkins-cli = handleTest ./jenkins-cli.nix {};
  jibri = handleTest ./jibri.nix {};
  jirafeau = handleTest ./jirafeau.nix {};
  jitsi-meet = handleTest ./jitsi-meet.nix {};
  jool = import ./jool.nix { inherit pkgs runTest; };
  jotta-cli = handleTest ./jotta-cli.nix {};
  k3s = handleTest ./k3s {};
  kafka = handleTest ./kafka.nix {};
  kanidm = handleTest ./kanidm.nix {};
  karma = handleTest ./karma.nix {};
  kavita = handleTest ./kavita.nix {};
  kbd-setfont-decompress = handleTest ./kbd-setfont-decompress.nix {};
  kbd-update-search-paths-patch = handleTest ./kbd-update-search-paths-patch.nix {};
  kea = handleTest ./kea.nix {};
  keepalived = handleTest ./keepalived.nix {};
  keepassxc = handleTest ./keepassxc.nix {};
  kerberos = handleTest ./kerberos/default.nix {};
  kernel-generic = handleTest ./kernel-generic.nix {};
  kernel-latest-ath-user-regd = handleTest ./kernel-latest-ath-user-regd.nix {};
  kernel-rust = handleTest ./kernel-rust.nix {};
  keter = handleTest ./keter.nix {};
  kexec = handleTest ./kexec.nix {};
  keycloak = discoverTests (import ./keycloak.nix);
  keyd = handleTest ./keyd.nix {};
  keymap = handleTest ./keymap.nix {};
  knot = handleTest ./knot.nix {};
  komga = handleTest ./komga.nix {};
  krb5 = discoverTests (import ./krb5);
  ksm = handleTest ./ksm.nix {};
  kthxbye = handleTest ./kthxbye.nix {};
  kubernetes = handleTestOn ["x86_64-linux"] ./kubernetes {};
  kubo = import ./kubo { inherit recurseIntoAttrs runTest; };
  ladybird = handleTest ./ladybird.nix {};
  languagetool = handleTest ./languagetool.nix {};
  lanraragi = handleTest ./lanraragi.nix {};
  latestKernel.login = handleTest ./login.nix { latestKernel = true; };
  leaps = handleTest ./leaps.nix {};
  lemmy = handleTest ./lemmy.nix {};
  libinput = handleTest ./libinput.nix {};
  libreddit = handleTest ./libreddit.nix {};
  librenms = handleTest ./librenms.nix {};
  libresprite = handleTest ./libresprite.nix {};
  libreswan = handleTest ./libreswan.nix {};
  librewolf = handleTest ./firefox.nix { firefoxPackage = pkgs.librewolf; };
  libuiohook = handleTest ./libuiohook.nix {};
  libvirtd = handleTest ./libvirtd.nix {};
  lidarr = handleTest ./lidarr.nix {};
  lightdm = handleTest ./lightdm.nix {};
  lighttpd = handleTest ./lighttpd.nix {};
  limesurvey = handleTest ./limesurvey.nix {};
  listmonk = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./listmonk.nix {};
  litestream = handleTest ./litestream.nix {};
  lldap = handleTest ./lldap.nix {};
  locate = handleTest ./locate.nix {};
  login = handleTest ./login.nix {};
  logrotate = handleTest ./logrotate.nix {};
  loki = handleTest ./loki.nix {};
  luks = handleTest ./luks.nix {};
  lvm2 = handleTest ./lvm2 {};
  lxd = pkgs.recurseIntoAttrs (handleTest ./lxd { inherit handleTestOn; });
  lxd-image-server = handleTest ./lxd-image-server.nix {};
  #logstash = handleTest ./logstash.nix {};
  lomiri = handleTest ./lomiri.nix {};
  lomiri-filemanager-app = runTest ./lomiri-filemanager-app.nix;
  lomiri-system-settings = handleTest ./lomiri-system-settings.nix {};
  lorri = handleTest ./lorri/default.nix {};
  maddy = discoverTests (import ./maddy { inherit handleTest; });
  maestral = handleTest ./maestral.nix {};
  magic-wormhole-mailbox-server = handleTest ./magic-wormhole-mailbox-server.nix {};
  magnetico = handleTest ./magnetico.nix {};
  mailcatcher = handleTest ./mailcatcher.nix {};
  mailhog = handleTest ./mailhog.nix {};
  mailman = handleTest ./mailman.nix {};
  man = handleTest ./man.nix {};
  mariadb-galera = handleTest ./mysql/mariadb-galera.nix {};
  mastodon = discoverTests (import ./web-apps/mastodon { inherit handleTestOn; });
  pixelfed = discoverTests (import ./web-apps/pixelfed { inherit handleTestOn; });
  mate = handleTest ./mate.nix {};
  mate-wayland = handleTest ./mate-wayland.nix {};
  matter-server = handleTest ./matter-server.nix {};
  matomo = handleTest ./matomo.nix {};
  matrix-appservice-irc = handleTest ./matrix/appservice-irc.nix {};
  matrix-conduit = handleTest ./matrix/conduit.nix {};
  matrix-synapse = handleTest ./matrix/synapse.nix {};
  matrix-synapse-workers = handleTest ./matrix/synapse-workers.nix {};
  mautrix-meta-postgres = handleTest ./matrix/mautrix-meta-postgres.nix {};
  mautrix-meta-sqlite = handleTest ./matrix/mautrix-meta-sqlite.nix {};
  mattermost = handleTest ./mattermost.nix {};
  mealie = handleTest ./mealie.nix {};
  mediamtx = handleTest ./mediamtx.nix {};
  mediatomb = handleTest ./mediatomb.nix {};
  mediawiki = handleTest ./mediawiki.nix {};
  meilisearch = handleTest ./meilisearch.nix {};
  memcached = handleTest ./memcached.nix {};
  merecat = handleTest ./merecat.nix {};
  metabase = handleTest ./metabase.nix {};
  mihomo = handleTest ./mihomo.nix {};
  mindustry = handleTest ./mindustry.nix {};
  minecraft = handleTest ./minecraft.nix {};
  minecraft-server = handleTest ./minecraft-server.nix {};
  minidlna = handleTest ./minidlna.nix {};
  miniflux = handleTest ./miniflux.nix {};
  minio = handleTest ./minio.nix {};
  miriway = handleTest ./miriway.nix {};
  misc = handleTest ./misc.nix {};
  mjolnir = handleTest ./matrix/mjolnir.nix {};
  mobilizon = handleTest ./mobilizon.nix {};
  mod_perl = handleTest ./mod_perl.nix {};
  molly-brown = handleTest ./molly-brown.nix {};
  mollysocket = handleTest ./mollysocket.nix { };
  monado = handleTest ./monado.nix {};
  monetdb = handleTest ./monetdb.nix {};
  monica = handleTest ./web-apps/monica.nix {};
  mongodb = handleTest ./mongodb.nix {};
  moodle = handleTest ./moodle.nix {};
  moonraker = handleTest ./moonraker.nix {};
  morph-browser = handleTest ./morph-browser.nix { };
  morty = handleTest ./morty.nix {};
  mosquitto = handleTest ./mosquitto.nix {};
  moosefs = handleTest ./moosefs.nix {};
  movim = discoverTests (import ./web-apps/movim { inherit handleTestOn; });
  mpd = handleTest ./mpd.nix {};
  mpv = handleTest ./mpv.nix {};
  mtp = handleTest ./mtp.nix {};
  multipass = handleTest ./multipass.nix {};
  mumble = handleTest ./mumble.nix {};
  # Fails on aarch64-linux at the PDF creation step - need to debug this on an
  # aarch64 machine..
  musescore = handleTestOn ["x86_64-linux"] ./musescore.nix {};
  munin = handleTest ./munin.nix {};
  mutableUsers = handleTest ./mutable-users.nix {};
  mxisd = handleTest ./mxisd.nix {};
  mycelium = handleTest ./mycelium {};
  mympd = handleTest ./mympd.nix {};
  mysql = handleTest ./mysql/mysql.nix {};
  mysql-autobackup = handleTest ./mysql/mysql-autobackup.nix {};
  mysql-backup = handleTest ./mysql/mysql-backup.nix {};
  mysql-replication = handleTest ./mysql/mysql-replication.nix {};
  n8n = handleTest ./n8n.nix {};
  nagios = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./nagios.nix {};
  nar-serve = handleTest ./nar-serve.nix {};
  nat.firewall = handleTest ./nat.nix { withFirewall = true; };
  nat.standalone = handleTest ./nat.nix { withFirewall = false; };
  nat.nftables.firewall = handleTest ./nat.nix { withFirewall = true; nftables = true; };
  nat.nftables.standalone = handleTest ./nat.nix { withFirewall = false; nftables = true; };
  nats = handleTest ./nats.nix {};
  navidrome = handleTest ./navidrome.nix {};
  nbd = handleTest ./nbd.nix {};
  ncdns = handleTest ./ncdns.nix {};
  ndppd = handleTest ./ndppd.nix {};
  nebula = handleTest ./nebula.nix {};
  netbird = handleTest ./netbird.nix {};
  nimdow = handleTest ./nimdow.nix {};
  neo4j = handleTest ./neo4j.nix {};
  netdata = handleTest ./netdata.nix {};
  networking.scripted = handleTest ./networking/networkd-and-scripted.nix { networkd = false; };
  networking.networkd = handleTest ./networking/networkd-and-scripted.nix { networkd = true; };
  networking.networkmanager = handleTest ./networking/networkmanager.nix {};
  netbox_3_6 = handleTest ./web-apps/netbox.nix { netbox = pkgs.netbox_3_6; };
  netbox_3_7 = handleTest ./web-apps/netbox.nix { netbox = pkgs.netbox_3_7; };
  netbox-upgrade = handleTest ./web-apps/netbox-upgrade.nix {};
  # TODO: put in networking.nix after the test becomes more complete
  networkingProxy = handleTest ./networking-proxy.nix {};
  nextcloud = handleTest ./nextcloud {};
  nexus = handleTest ./nexus.nix {};
  # TODO: Test nfsv3 + Kerberos
  nfs3 = handleTest ./nfs { version = 3; };
  nfs4 = handleTest ./nfs { version = 4; };
  nghttpx = handleTest ./nghttpx.nix {};
  nginx = handleTest ./nginx.nix {};
  nginx-auth = handleTest ./nginx-auth.nix {};
  nginx-etag = handleTest ./nginx-etag.nix {};
  nginx-etag-compression = handleTest ./nginx-etag-compression.nix {};
  nginx-globalredirect = handleTest ./nginx-globalredirect.nix {};
  nginx-http3 = handleTest ./nginx-http3.nix {};
  nginx-modsecurity = handleTest ./nginx-modsecurity.nix {};
  nginx-moreheaders = handleTest ./nginx-moreheaders.nix {};
  nginx-njs = handleTest ./nginx-njs.nix {};
  nginx-proxyprotocol = handleTest ./nginx-proxyprotocol {};
  nginx-pubhtml = handleTest ./nginx-pubhtml.nix {};
  nginx-redirectcode = handleTest ./nginx-redirectcode.nix {};
  nginx-sso = handleTest ./nginx-sso.nix {};
  nginx-status-page = handleTest ./nginx-status-page.nix {};
  nginx-tmpdir = handleTest ./nginx-tmpdir.nix {};
  nginx-unix-socket = handleTest ./nginx-unix-socket.nix {};
  nginx-variants = handleTest ./nginx-variants.nix {};
  nifi = handleTestOn ["x86_64-linux"] ./web-apps/nifi.nix {};
  nitter = handleTest ./nitter.nix {};
  nix-config = handleTest ./nix-config.nix {};
  nix-ld = handleTest ./nix-ld.nix {};
  nix-serve = handleTest ./nix-serve.nix {};
  nix-serve-ssh = handleTest ./nix-serve-ssh.nix {};
  nixops = handleTest ./nixops/default.nix {};
  nixos-generate-config = handleTest ./nixos-generate-config.nix {};
  nixos-rebuild-install-bootloader = handleTestOn ["x86_64-linux"] ./nixos-rebuild-install-bootloader.nix {};
  nixos-rebuild-specialisations = handleTestOn ["x86_64-linux"] ./nixos-rebuild-specialisations.nix {};
  nixos-rebuild-target-host = handleTest ./nixos-rebuild-target-host.nix {};
  nixpkgs = pkgs.callPackage ../modules/misc/nixpkgs/test.nix { inherit evalMinimalConfig; };
  nixseparatedebuginfod = handleTest ./nixseparatedebuginfod.nix {};
  node-red = handleTest ./node-red.nix {};
  nomad = handleTest ./nomad.nix {};
  non-default-filesystems = handleTest ./non-default-filesystems.nix {};
  non-switchable-system = runTest ./non-switchable-system.nix;
  noto-fonts = handleTest ./noto-fonts.nix {};
  noto-fonts-cjk-qt-default-weight = handleTest ./noto-fonts-cjk-qt-default-weight.nix {};
  novacomd = handleTestOn ["x86_64-linux"] ./novacomd.nix {};
  npmrc = handleTest ./npmrc.nix {};
  nscd = handleTest ./nscd.nix {};
  nsd = handleTest ./nsd.nix {};
  ntfy-sh = handleTest ./ntfy-sh.nix {};
  ntfy-sh-migration = handleTest ./ntfy-sh-migration.nix {};
  ntpd-rs = handleTest ./ntpd-rs.nix {};
  nvmetcfg = handleTest ./nvmetcfg.nix {};
  nzbget = handleTest ./nzbget.nix {};
  nzbhydra2 = handleTest ./nzbhydra2.nix {};
  ocis = handleTest ./ocis.nix {};
  oddjobd = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./oddjobd.nix {};
  oh-my-zsh = handleTest ./oh-my-zsh.nix {};
  ollama = handleTest ./ollama.nix {};
  ombi = handleTest ./ombi.nix {};
  openarena = handleTest ./openarena.nix {};
  openldap = handleTest ./openldap.nix {};
  opensearch = discoverTests (import ./opensearch.nix);
  openresty-lua = handleTest ./openresty-lua.nix {};
  opensmtpd = handleTest ./opensmtpd.nix {};
  opensmtpd-rspamd = handleTest ./opensmtpd-rspamd.nix {};
  opensnitch = handleTest ./opensnitch.nix {};
  openssh = handleTest ./openssh.nix {};
  octoprint = handleTest ./octoprint.nix {};
  openstack-image-metadata = (handleTestOn ["x86_64-linux"] ./openstack-image.nix {}).metadata or {};
  openstack-image-userdata = (handleTestOn ["x86_64-linux"] ./openstack-image.nix {}).userdata or {};
  opentabletdriver = handleTest ./opentabletdriver.nix {};
  opentelemetry-collector = handleTest ./opentelemetry-collector.nix {};
  ocsinventory-agent = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./ocsinventory-agent.nix {};
  owncast = handleTest ./owncast.nix {};
  outline = handleTest ./outline.nix {};
  image-contents = handleTest ./image-contents.nix {};
  openvscode-server = handleTest ./openvscode-server.nix {};
  orangefs = handleTest ./orangefs.nix {};
  os-prober = handleTestOn ["x86_64-linux"] ./os-prober.nix {};
  osquery = handleTestOn ["x86_64-linux"] ./osquery.nix {};
  osrm-backend = handleTest ./osrm-backend.nix {};
  overlayfs = handleTest ./overlayfs.nix {};
  pacemaker = handleTest ./pacemaker.nix {};
  packagekit = handleTest ./packagekit.nix {};
  pam-file-contents = handleTest ./pam/pam-file-contents.nix {};
  pam-oath-login = handleTest ./pam/pam-oath-login.nix {};
  pam-u2f = handleTest ./pam/pam-u2f.nix {};
  pam-ussh = handleTest ./pam/pam-ussh.nix {};
  pam-zfs-key = handleTest ./pam/zfs-key.nix {};
  pass-secret-service = handleTest ./pass-secret-service.nix {};
  patroni = handleTestOn ["x86_64-linux"] ./patroni.nix {};
  pantalaimon = handleTest ./matrix/pantalaimon.nix {};
  pantheon = handleTest ./pantheon.nix {};
  paperless = handleTest ./paperless.nix {};
  parsedmarc = handleTest ./parsedmarc {};
  pdns-recursor = handleTest ./pdns-recursor.nix {};
  peerflix = handleTest ./peerflix.nix {};
  peering-manager = handleTest ./web-apps/peering-manager.nix {};
  peertube = handleTestOn ["x86_64-linux"] ./web-apps/peertube.nix {};
  peroxide = handleTest ./peroxide.nix {};
  pg_anonymizer = handleTest ./pg_anonymizer.nix {};
  pgadmin4 = handleTest ./pgadmin4.nix {};
  pgbouncer = handleTest ./pgbouncer.nix {};
  pgjwt = handleTest ./pgjwt.nix {};
  pgmanage = handleTest ./pgmanage.nix {};
  pgvecto-rs = handleTest ./pgvecto-rs.nix {};
  phosh = handleTest ./phosh.nix {};
  photonvision = handleTest ./photonvision.nix {};
  photoprism = handleTest ./photoprism.nix {};
  php = handleTest ./php {};
  php81 = handleTest ./php { php = pkgs.php81; };
  php82 = handleTest ./php { php = pkgs.php82; };
  php83 = handleTest ./php { php = pkgs.php83; };
  phylactery = handleTest ./web-apps/phylactery.nix {};
  pict-rs = handleTest ./pict-rs.nix {};
  pinnwand = handleTest ./pinnwand.nix {};
  plantuml-server = handleTest ./plantuml-server.nix {};
  plasma-bigscreen = handleTest ./plasma-bigscreen.nix {};
  plasma5 = handleTest ./plasma5.nix {};
  plasma6 = handleTest ./plasma6.nix {};
  plasma5-systemd-start = handleTest ./plasma5-systemd-start.nix {};
  plausible = handleTest ./plausible.nix {};
  please = handleTest ./please.nix {};
  pleroma = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./pleroma.nix {};
  plikd = handleTest ./plikd.nix {};
  plotinus = handleTest ./plotinus.nix {};
  podgrab = handleTest ./podgrab.nix {};
  podman = handleTestOn ["aarch64-linux" "x86_64-linux"] ./podman/default.nix {};
  podman-tls-ghostunnel = handleTestOn ["aarch64-linux" "x86_64-linux"] ./podman/tls-ghostunnel.nix {};
  polaris = handleTest ./polaris.nix {};
  pomerium = handleTestOn ["x86_64-linux"] ./pomerium.nix {};
  portunus = handleTest ./portunus.nix { };
  postfix = handleTest ./postfix.nix {};
  postfix-raise-smtpd-tls-security-level = handleTest ./postfix-raise-smtpd-tls-security-level.nix {};
  postfixadmin = handleTest ./postfixadmin.nix {};
  postgis = handleTest ./postgis.nix {};
  apache_datasketches = handleTest ./apache_datasketches.nix {};
  postgresql = handleTest ./postgresql.nix {};
  postgresql-jit = handleTest ./postgresql-jit.nix {};
  postgresql-wal-receiver = handleTest ./postgresql-wal-receiver.nix {};
  powerdns = handleTest ./powerdns.nix {};
  powerdns-admin = handleTest ./powerdns-admin.nix {};
  power-profiles-daemon = handleTest ./power-profiles-daemon.nix {};
  pppd = handleTest ./pppd.nix {};
  predictable-interface-names = handleTest ./predictable-interface-names.nix {};
  pretalx = runTest ./web-apps/pretalx.nix;
  pretix = runTest ./web-apps/pretix.nix;
  printing-socket = handleTest ./printing.nix { socket = true; };
  printing-service = handleTest ./printing.nix { socket = false; };
  private-gpt = handleTest ./private-gpt.nix {};
  privoxy = handleTest ./privoxy.nix {};
  prometheus = handleTest ./prometheus.nix {};
  prometheus-exporters = handleTest ./prometheus-exporters.nix {};
  prosody = handleTest ./xmpp/prosody.nix {};
  prosody-mysql = handleTest ./xmpp/prosody-mysql.nix {};
  proxy = handleTest ./proxy.nix {};
  prowlarr = handleTest ./prowlarr.nix {};
  pt2-clone = handleTest ./pt2-clone.nix {};
  pykms = handleTest ./pykms.nix {};
  public-inbox = handleTest ./public-inbox.nix {};
  pufferpanel = handleTest ./pufferpanel.nix {};
  pulseaudio = discoverTests (import ./pulseaudio.nix);
  qboot = handleTestOn ["x86_64-linux" "i686-linux"] ./qboot.nix {};
  qemu-vm-restrictnetwork = handleTest ./qemu-vm-restrictnetwork.nix {};
  qemu-vm-volatile-root = runTest ./qemu-vm-volatile-root.nix;
  qemu-vm-external-disk-image = runTest ./qemu-vm-external-disk-image.nix;
  qgis = handleTest ./qgis.nix { qgisPackage = pkgs.qgis; };
  qgis-ltr = handleTest ./qgis.nix { qgisPackage = pkgs.qgis-ltr; };
  qownnotes = handleTest ./qownnotes.nix {};
  qtile = handleTest ./qtile.nix {};
  quake3 = handleTest ./quake3.nix {};
  quicktun = handleTest ./quicktun.nix {};
  quorum = handleTest ./quorum.nix {};
  rabbitmq = handleTest ./rabbitmq.nix {};
  radarr = handleTest ./radarr.nix {};
  radicale = handleTest ./radicale.nix {};
  ragnarwm = handleTest ./ragnarwm.nix {};
  rasdaemon = handleTest ./rasdaemon.nix {};
  readarr = handleTest ./readarr.nix {};
  redis = handleTest ./redis.nix {};
  redlib = handleTest ./redlib.nix {};
  redmine = handleTest ./redmine.nix {};
  restartByActivationScript = handleTest ./restart-by-activation-script.nix {};
  restic-rest-server = handleTest ./restic-rest-server.nix {};
  restic = handleTest ./restic.nix {};
  retroarch = handleTest ./retroarch.nix {};
  rkvm = handleTest ./rkvm {};
  robustirc-bridge = handleTest ./robustirc-bridge.nix {};
  roundcube = handleTest ./roundcube.nix {};
  rosenpass = handleTest ./rosenpass.nix {};
  rshim = handleTest ./rshim.nix {};
  rspamd = handleTest ./rspamd.nix {};
  rspamd-trainer = handleTest ./rspamd-trainer.nix {};
  rss2email = handleTest ./rss2email.nix {};
  rstudio-server = handleTest ./rstudio-server.nix {};
  rsyncd = handleTest ./rsyncd.nix {};
  rsyslogd = handleTest ./rsyslogd.nix {};
  rxe = handleTest ./rxe.nix {};
  sabnzbd = handleTest ./sabnzbd.nix {};
  samba = handleTest ./samba.nix {};
  samba-wsdd = handleTest ./samba-wsdd.nix {};
  sane = handleTest ./sane.nix {};
  sanoid = handleTest ./sanoid.nix {};
  scaphandre = handleTest ./scaphandre.nix {};
  schleuder = handleTest ./schleuder.nix {};
  scion-freestanding-deployment = handleTest ./scion/freestanding-deployment {};
  scrutiny = handleTest ./scrutiny.nix {};
  sddm = handleTest ./sddm.nix {};
  seafile = handleTest ./seafile.nix {};
  searx = handleTest ./searx.nix {};
  seatd = handleTest ./seatd.nix {};
  service-runner = handleTest ./service-runner.nix {};
  sftpgo = runTest ./sftpgo.nix;
  sfxr-qt = handleTest ./sfxr-qt.nix {};
  sgt-puzzles = handleTest ./sgt-puzzles.nix {};
  shadow = handleTest ./shadow.nix {};
  shadowsocks = handleTest ./shadowsocks {};
  shattered-pixel-dungeon = handleTest ./shattered-pixel-dungeon.nix {};
  shiori = handleTest ./shiori.nix {};
  signal-desktop = handleTest ./signal-desktop.nix {};
  silverbullet = handleTest ./silverbullet.nix {};
  simple = handleTest ./simple.nix {};
  sing-box = handleTest ./sing-box.nix {};
  slimserver = handleTest ./slimserver.nix {};
  slurm = handleTest ./slurm.nix {};
  snmpd = handleTest ./snmpd.nix {};
  smokeping = handleTest ./smokeping.nix {};
  snapcast = handleTest ./snapcast.nix {};
  snapper = handleTest ./snapper.nix {};
  snipe-it = runTest ./web-apps/snipe-it.nix;
  soapui = handleTest ./soapui.nix {};
  soft-serve = handleTest ./soft-serve.nix {};
  sogo = handleTest ./sogo.nix {};
  soju = handleTest ./soju.nix {};
  solanum = handleTest ./solanum.nix {};
  sonarr = handleTest ./sonarr.nix {};
  sonic-server = handleTest ./sonic-server.nix {};
  sourcehut = handleTest ./sourcehut {};
  spacecookie = handleTest ./spacecookie.nix {};
  spark = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./spark {};
  sqlite3-to-mysql = handleTest ./sqlite3-to-mysql.nix {};
  sslh = handleTest ./sslh.nix {};
  ssh-agent-auth = handleTest ./ssh-agent-auth.nix {};
  ssh-audit = handleTest ./ssh-audit.nix {};
  sssd = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./sssd.nix {};
  sssd-ldap = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./sssd-ldap.nix {};
  stalwart-mail = handleTest ./stalwart-mail.nix {};
  stargazer = runTest ./web-servers/stargazer.nix;
  starship = handleTest ./starship.nix {};
  static-web-server = handleTest ./web-servers/static-web-server.nix {};
  step-ca = handleTestOn ["x86_64-linux"] ./step-ca.nix {};
  stratis = handleTest ./stratis {};
  strongswan-swanctl = handleTest ./strongswan-swanctl.nix {};
  stub-ld = handleTestOn [ "x86_64-linux" "aarch64-linux" ] ./stub-ld.nix {};
  stunnel = handleTest ./stunnel.nix {};
  sudo = handleTest ./sudo.nix {};
  sudo-rs = handleTest ./sudo-rs.nix {};
  sunshine = handleTest ./sunshine.nix {};
  suwayomi-server = handleTest ./suwayomi-server.nix {};
  swap-file-btrfs = handleTest ./swap-file-btrfs.nix {};
  swap-partition = handleTest ./swap-partition.nix {};
  swap-random-encryption = handleTest ./swap-random-encryption.nix {};
  sway = handleTest ./sway.nix {};
  swayfx = handleTest ./swayfx.nix {};
  switchTest = handleTest ./switch-test.nix { ng = false; };
  switchTestNg = handleTest ./switch-test.nix { ng = true; };
  sympa = handleTest ./sympa.nix {};
  syncthing = handleTest ./syncthing.nix {};
  syncthing-no-settings = handleTest ./syncthing-no-settings.nix {};
  syncthing-init = handleTest ./syncthing-init.nix {};
  syncthing-many-devices = handleTest ./syncthing-many-devices.nix {};
  syncthing-relay = handleTest ./syncthing-relay.nix {};
  sysinit-reactivation = runTest ./sysinit-reactivation.nix;
  systemd = handleTest ./systemd.nix {};
  systemd-analyze = handleTest ./systemd-analyze.nix {};
  systemd-binfmt = handleTestOn ["x86_64-linux"] ./systemd-binfmt.nix {};
  systemd-boot = handleTest ./systemd-boot.nix {};
  systemd-bpf = handleTest ./systemd-bpf.nix {};
  systemd-confinement = handleTest ./systemd-confinement {};
  systemd-coredump = handleTest ./systemd-coredump.nix {};
  systemd-cryptenroll = handleTest ./systemd-cryptenroll.nix {};
  systemd-credentials-tpm2 = handleTest ./systemd-credentials-tpm2.nix {};
  systemd-escaping = handleTest ./systemd-escaping.nix {};
  systemd-initrd-bridge = handleTest ./systemd-initrd-bridge.nix {};
  systemd-initrd-btrfs-raid = handleTest ./systemd-initrd-btrfs-raid.nix {};
  systemd-initrd-luks-fido2 = handleTest ./systemd-initrd-luks-fido2.nix {};
  systemd-initrd-luks-keyfile = handleTest ./systemd-initrd-luks-keyfile.nix {};
  systemd-initrd-luks-empty-passphrase = handleTest ./initrd-luks-empty-passphrase.nix { systemdStage1 = true; };
  systemd-initrd-luks-password = handleTest ./systemd-initrd-luks-password.nix {};
  systemd-initrd-luks-tpm2 = handleTest ./systemd-initrd-luks-tpm2.nix {};
  systemd-initrd-luks-unl0kr = handleTest ./systemd-initrd-luks-unl0kr.nix {};
  systemd-initrd-modprobe = handleTest ./systemd-initrd-modprobe.nix {};
  systemd-initrd-shutdown = handleTest ./systemd-shutdown.nix { systemdStage1 = true; };
  systemd-initrd-simple = handleTest ./systemd-initrd-simple.nix {};
  systemd-initrd-swraid = handleTest ./systemd-initrd-swraid.nix {};
  systemd-initrd-vconsole = handleTest ./systemd-initrd-vconsole.nix {};
  systemd-initrd-networkd = handleTest ./systemd-initrd-networkd.nix {};
  systemd-initrd-networkd-ssh = handleTest ./systemd-initrd-networkd-ssh.nix {};
  systemd-initrd-networkd-openvpn = handleTestOn [ "x86_64-linux" "i686-linux" ] ./initrd-network-openvpn { systemdStage1 = true; };
  systemd-initrd-vlan = handleTest ./systemd-initrd-vlan.nix {};
  systemd-journal = handleTest ./systemd-journal.nix {};
  systemd-journal-gateway = handleTest ./systemd-journal-gateway.nix {};
  systemd-journal-upload = handleTest ./systemd-journal-upload.nix {};
  systemd-lock-handler = runTestOn ["aarch64-linux" "x86_64-linux"] ./systemd-lock-handler.nix;
  systemd-machinectl = handleTest ./systemd-machinectl.nix {};
  systemd-networkd = handleTest ./systemd-networkd.nix {};
  systemd-networkd-bridge = handleTest ./systemd-networkd-bridge.nix {};
  systemd-networkd-dhcpserver = handleTest ./systemd-networkd-dhcpserver.nix {};
  systemd-networkd-dhcpserver-static-leases = handleTest ./systemd-networkd-dhcpserver-static-leases.nix {};
  systemd-networkd-ipv6-prefix-delegation = handleTest ./systemd-networkd-ipv6-prefix-delegation.nix {};
  systemd-networkd-vrf = handleTest ./systemd-networkd-vrf.nix {};
  systemd-no-tainted = handleTest ./systemd-no-tainted.nix {};
  systemd-nspawn = handleTest ./systemd-nspawn.nix {};
  systemd-nspawn-configfile = handleTest ./systemd-nspawn-configfile.nix {};
  systemd-oomd = handleTest ./systemd-oomd.nix {};
  systemd-portabled = handleTest ./systemd-portabled.nix {};
  systemd-repart = handleTest ./systemd-repart.nix {};
  systemd-shutdown = handleTest ./systemd-shutdown.nix {};
  systemd-sysupdate = runTest ./systemd-sysupdate.nix;
  systemd-sysusers-mutable = runTest ./systemd-sysusers-mutable.nix;
  systemd-sysusers-immutable = runTest ./systemd-sysusers-immutable.nix;
  systemd-timesyncd = handleTest ./systemd-timesyncd.nix {};
  systemd-timesyncd-nscd-dnssec = handleTest ./systemd-timesyncd-nscd-dnssec.nix {};
  systemd-user-linger = handleTest ./systemd-user-linger.nix {};
  systemd-user-tmpfiles-rules = handleTest ./systemd-user-tmpfiles-rules.nix {};
  systemd-misc = handleTest ./systemd-misc.nix {};
  systemd-userdbd = handleTest ./systemd-userdbd.nix {};
  systemd-homed = handleTest ./systemd-homed.nix {};
  systemtap = handleTest ./systemtap.nix {};
  tandoor-recipes = handleTest ./tandoor-recipes.nix {};
  tang = handleTest ./tang.nix {};
  taskserver = handleTest ./taskserver.nix {};
  tayga = handleTest ./tayga.nix {};
  technitium-dns-server = handleTest ./technitium-dns-server.nix {};
  teeworlds = handleTest ./teeworlds.nix {};
  telegraf = handleTest ./telegraf.nix {};
  teleport = handleTest ./teleport.nix {};
  thelounge = handleTest ./thelounge.nix {};
  terminal-emulators = handleTest ./terminal-emulators.nix {};
  tiddlywiki = handleTest ./tiddlywiki.nix {};
  tigervnc = handleTest ./tigervnc.nix {};
  timescaledb = handleTest ./timescaledb.nix {};
  timezone = handleTest ./timezone.nix {};
  tinc = handleTest ./tinc {};
  tinydns = handleTest ./tinydns.nix {};
  tinyproxy = handleTest ./tinyproxy.nix {};
  tinywl = handleTest ./tinywl.nix {};
  tmate-ssh-server = handleTest ./tmate-ssh-server.nix { };
  tomcat = handleTest ./tomcat.nix {};
  tor = handleTest ./tor.nix {};
  traefik = handleTestOn ["aarch64-linux" "x86_64-linux"] ./traefik.nix {};
  trafficserver = handleTest ./trafficserver.nix {};
  transfer-sh = handleTest ./transfer-sh.nix {};
  transmission = handleTest ./transmission.nix { transmission = pkgs.transmission; };
  transmission_4 = handleTest ./transmission.nix { transmission = pkgs.transmission_4; };
  # tracee requires bpf
  tracee = handleTestOn ["x86_64-linux"] ./tracee.nix {};
  trezord = handleTest ./trezord.nix {};
  trickster = handleTest ./trickster.nix {};
  trilium-server = handleTestOn ["x86_64-linux"] ./trilium-server.nix {};
  tsja = handleTest ./tsja.nix {};
  tsm-client-gui = handleTest ./tsm-client-gui.nix {};
  ttyd = handleTest ./web-servers/ttyd.nix {};
  txredisapi = handleTest ./txredisapi.nix {};
  tuptime = handleTest ./tuptime.nix {};
  turbovnc-headless-server = handleTest ./turbovnc-headless-server.nix {};
  tuxguitar = handleTest ./tuxguitar.nix {};
  twingate = runTest ./twingate.nix;
  typesense = handleTest ./typesense.nix {};
  ucarp = handleTest ./ucarp.nix {};
  udisks2 = handleTest ./udisks2.nix {};
  ulogd = handleTest ./ulogd/ulogd.nix {};
  unbound = handleTest ./unbound.nix {};
  unifi = handleTest ./unifi.nix {};
  unit-php = handleTest ./web-servers/unit-php.nix {};
  upnp.iptables = handleTest ./upnp.nix { useNftables = false; };
  upnp.nftables = handleTest ./upnp.nix { useNftables = true; };
  uptermd = handleTest ./uptermd.nix {};
  uptime-kuma = handleTest ./uptime-kuma.nix {};
  urn-timer = handleTest ./urn-timer.nix {};
  usbguard = handleTest ./usbguard.nix {};
  user-activation-scripts = handleTest ./user-activation-scripts.nix {};
  user-expiry = runTest ./user-expiry.nix;
  user-home-mode = handleTest ./user-home-mode.nix {};
  ustreamer = handleTest ./ustreamer.nix {};
  uwsgi = handleTest ./uwsgi.nix {};
  v2ray = handleTest ./v2ray.nix {};
  varnish60 = handleTest ./varnish.nix { package = pkgs.varnish60; };
  varnish74 = handleTest ./varnish.nix { package = pkgs.varnish74; };
  varnish75 = handleTest ./varnish.nix { package = pkgs.varnish75; };
  vault = handleTest ./vault.nix {};
  vault-agent = handleTest ./vault-agent.nix {};
  vault-dev = handleTest ./vault-dev.nix {};
  vault-postgresql = handleTest ./vault-postgresql.nix {};
  vaultwarden = handleTest ./vaultwarden.nix {};
  vector = handleTest ./vector {};
  vengi-tools = handleTest ./vengi-tools.nix {};
  victoriametrics = handleTest ./victoriametrics.nix {};
  vikunja = handleTest ./vikunja.nix {};
  virtualbox = handleTestOn ["x86_64-linux"] ./virtualbox.nix {};
  vscode-remote-ssh = handleTestOn ["x86_64-linux"] ./vscode-remote-ssh.nix {};
  vscodium = discoverTests (import ./vscodium.nix);
  vsftpd = handleTest ./vsftpd.nix {};
  warzone2100 = handleTest ./warzone2100.nix {};
  wasabibackend = handleTest ./wasabibackend.nix {};
  wastebin = handleTest ./wastebin.nix {};
  watchdogd = handleTest ./watchdogd.nix {};
  webhook = runTest ./webhook.nix;
  wiki-js = handleTest ./wiki-js.nix {};
  wine = handleTest ./wine.nix {};
  wireguard = handleTest ./wireguard {};
  without-nix = handleTest ./without-nix.nix {};
  wmderland = handleTest ./wmderland.nix {};
  workout-tracker = handleTest ./workout-tracker.nix {};
  wpa_supplicant = handleTest ./wpa_supplicant.nix {};
  wordpress = handleTest ./wordpress.nix {};
  wrappers = handleTest ./wrappers.nix {};
  writefreely = handleTest ./web-apps/writefreely.nix {};
  xandikos = handleTest ./xandikos.nix {};
  xautolock = handleTest ./xautolock.nix {};
  xfce = handleTest ./xfce.nix {};
  xmonad = handleTest ./xmonad.nix {};
  xmonad-xdg-autostart = handleTest ./xmonad-xdg-autostart.nix {};
  xpadneo = handleTest ./xpadneo.nix {};
  xrdp = handleTest ./xrdp.nix {};
  xrdp-with-audio-pulseaudio = handleTest ./xrdp-with-audio-pulseaudio.nix {};
  xscreensaver = handleTest ./xscreensaver.nix {};
  xss-lock = handleTest ./xss-lock.nix {};
  xterm = handleTest ./xterm.nix {};
  xxh = handleTest ./xxh.nix {};
  yabar = handleTest ./yabar.nix {};
  ydotool = handleTest ./ydotool.nix {};
  yggdrasil = handleTest ./yggdrasil.nix {};
  your_spotify = handleTest ./your_spotify.nix {};
  zammad = handleTest ./zammad.nix {};
  zeronet-conservancy = handleTest ./zeronet-conservancy.nix {};
  zfs = handleTest ./zfs.nix {};
  zigbee2mqtt = handleTest ./zigbee2mqtt.nix {};
  zoneminder = handleTest ./zoneminder.nix {};
  zookeeper = handleTest ./zookeeper.nix {};
  zram-generator = handleTest ./zram-generator.nix {};
  zrepl = handleTest ./zrepl.nix {};
  zsh-history = handleTest ./zsh-history.nix {};
  zwave-js = handleTest ./zwave-js.nix {};
}
