#!/usr/bin/env nix-shell
#! nix-shell -i bash -p bundix coreutils diffutils nix-prefetch-github gnused jq prefetch-yarn-deps
set -e

OWNER=mastodon
REPO=mastodon

POSITIONAL=()
while [[ $# -gt 0 ]]; do
    key="$1"

    case $key in
        --owner)
            OWNER="$2"
            shift # past argument
            shift # past value
            ;;
        --repo)
            REPO="$2"
            shift # past argument
            shift # past value
            ;;
        --ver)
            VERSION="$2"
            shift # past argument
            shift # past value
            ;;
        --rev)
            REVISION="$2"
            shift # past argument
            shift # past value
            ;;
        --patches)
            PATCHES="$2"
            shift # past argument
            shift # past value
            ;;
        *)  # unknown option
            POSITIONAL+=("$1")
            shift # past argument
            ;;
    esac
done

if [[ -n "$POSITIONAL" ]]; then
    echo "Usage: update.sh [--owner OWNER] [--repo REPO] [--ver VERSION] [--rev REVISION] [--patches PATCHES]"
    echo "OWNER and REPO must be paths on github."
    echo "If REVISION is not provided, the latest tag from github.com/mastodon/mastodon is fetched and VERSION is calculated from it."
    echo "If OWNER and REPO are not provided, it defaults they default to mastodon and mastodon."
    echo "PATCHES, if provided, should be one or more Nix expressions separated by spaces."
    exit 1
fi

if [[ -z "$REVISION" ]]; then
    REVISION="$(curl ${GITHUB_TOKEN:+" -u \":$GITHUB_TOKEN\""} -s "https://api.github.com/repos/$OWNER/$REPO/releases" | jq -r  'map(select(.prerelease == false)) | .[0].tag_name')"
fi

VERSION="$(echo "$REVISION" | cut -c2-)"

rm -f gemset.nix source.nix
cd "$(dirname "${BASH_SOURCE[0]}")" || exit 1

WORK_DIR=$(mktemp -d)

# Check that working directory was created.
if [[ -z "$WORK_DIR" || ! -d "$WORK_DIR" ]]; then
    echo "Could not create temporary directory"
    exit 1
fi

# Delete the working directory on exit.
function cleanup {
    # Report errors, if any, from nix-prefetch-git
    grep "fatal" $WORK_DIR/nix-prefetch-git.out >/dev/stderr || true
    rm -rf "$WORK_DIR"
}
trap cleanup EXIT

echo "Fetching source code $REVISION"
JSON=$(nix-prefetch-github "$OWNER" "$REPO" --rev "$REVISION" 2> $WORK_DIR/nix-prefetch-git.out)
HASH=$(echo "$JSON" | jq -r .hash)

cat > source.nix << EOF
# This file was generated by pkgs.mastodon.updateScript.
{ fetchFromGitHub, applyPatches, patches ? [] }:
let
  version = "$VERSION";
in
(
  applyPatches {
    src = fetchFromGitHub {
      owner = "$OWNER";
      repo = "$REPO";
      rev = "v\${version}";
      hash = "$HASH";
    };
    patches = patches ++ [$PATCHES];
  }) // {
  inherit version;
  yarnHash = "sha256-AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA=";
}
EOF
SOURCE_DIR="$(nix-build --no-out-link -E '(import <nixpkgs> {}).callPackage ./source.nix {}')"

echo "Creating gemset.nix"
bundix --lockfile="$SOURCE_DIR/Gemfile.lock" --gemfile="$SOURCE_DIR/Gemfile"
echo "" >> gemset.nix  # Create trailing newline to please EditorConfig checks

echo "Creating yarn-hash.nix"
YARN_HASH="$(prefetch-yarn-deps "$SOURCE_DIR/yarn.lock")"
YARN_HASH="$(nix hash to-sri --type sha256 "$YARN_HASH")"
sed -i "s/sha256-AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA=/$YARN_HASH/g" source.nix
