#!/usr/bin/env python

# run with:
# $ nix run .\#vimPluginsUpdater
# format:
# $ nix run nixpkgs#python3Packages.black -- update.py
# type-check:
# $ nix run nixpkgs#python3Packages.mypy -- update.py
# linted:
# $ nix run nixpkgs#python3Packages.flake8 -- --ignore E501,E265,E402 update.py

# If you see `HTTP Error 429: too many requests` errors while running this
# script, refer to:
#
# https://github.com/NixOS/nixpkgs/blob/master/doc/languages-frameworks/vim.section.md#updating-plugins-in-nixpkgs-updating-plugins-in-nixpkgs
#
# (or the equivalent file /doc/languages-frameworks/vim.section.md
# from Nixpkgs master tree).
#

import inspect
import os
import logging
import textwrap
import json
import subprocess
from typing import List, Tuple
from pathlib import Path


log = logging.getLogger("vim-updater")

sh = logging.StreamHandler()
formatter = logging.Formatter("%(name)s:%(levelname)s: %(message)s")
sh.setFormatter(formatter)
log.addHandler(sh)

# Import plugin update library from maintainers/scripts/pluginupdate.py
ROOT = Path(os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe()))))
import pluginupdate
import importlib
from pluginupdate import run_nix_expr, PluginDesc
import treesitter


HEADER = (
    "# GENERATED by ./pkgs/applications/editors/vim/plugins/update.py. Do not edit!"
)

NIXPKGS_NVIMTREESITTER_FOLDER = "pkgs/applications/editors/vim/plugins/nvim-treesitter"


class VimEditor(pluginupdate.Editor):
    nvim_treesitter_updated = False

    def generate_nix(
        self, plugins: List[Tuple[PluginDesc, pluginupdate.Plugin]], outfile: str
    ):
        sorted_plugins = sorted(plugins, key=lambda v: v[0].name.lower())
        nvim_treesitter_rev = pluginupdate.run_nix_expr(
            "(import <localpkgs> { }).vimPlugins.nvim-treesitter.src.rev", self.nixpkgs
        )

        with open(outfile, "w+") as f:
            f.write(HEADER)
            f.write(
                textwrap.dedent(
                    """
                { lib, buildVimPlugin, buildNeovimPlugin, fetchFromGitHub, fetchgit }:

                final: prev:
                {
                """
                )
            )
            for pdesc, plugin in sorted_plugins:
                content = self.plugin2nix(pdesc, plugin)
                f.write(content)
                if (
                    plugin.name == "nvim-treesitter"
                    and plugin.commit != nvim_treesitter_rev
                ):
                    self.nvim_treesitter_updated = True
            f.write("\n}\n")
        print(f"updated {outfile}")

    def plugin2nix(self, pdesc: PluginDesc, plugin: pluginupdate.Plugin) -> str:
        GET_PLUGINS_LUA = """
        with import <localpkgs> {};
        lib.attrNames lua51Packages"""
        luaPlugins = run_nix_expr(GET_PLUGINS_LUA, self.nixpkgs)

        repo = pdesc.repo

        def _isNeovimPlugin(plug: pluginupdate.Plugin) -> bool:
            """
            Whether it's a neovim-only plugin
            We can check if it's available in lua packages
            """
            # global luaPlugins
            if plug.normalized_name in luaPlugins:
                log.debug("%s is a neovim plugin", plug)
                return True
            return False

        isNeovim = _isNeovimPlugin(plugin)

        content = f"  {plugin.normalized_name} = "
        src_nix = repo.as_nix(plugin)
        content += """{buildFn} {{
    pname = "{plugin.name}";
    version = "{plugin.version}";
    src = {src_nix};
    meta.homepage = "{repo.uri}";
  }};

""".format(
            buildFn="buildNeovimPlugin" if isNeovim else "buildVimPlugin",
            plugin=plugin,
            src_nix=src_nix,
            repo=repo,
        )
        log.debug(content)
        return content

    def update(self, args):
        pluginupdate.update_plugins(self, args)

        # TODO this should probably be skipped when running outside a nixpkgs checkout
        if self.nvim_treesitter_updated:
            print("updating nvim-treesitter grammars")
            cmd = [
                "nix", "build",
                "vimPlugins.nvim-treesitter.src", "-f", self.nixpkgs
                , "--print-out-paths"
            ]
            log.debug("Running command: %s", " ".join(cmd))
            nvim_treesitter_dir = subprocess.check_output(cmd, text=True, timeout=90).strip()

            generated = treesitter.update_grammars(nvim_treesitter_dir)
            treesitter_generated_nix_path = os.path.join(
                NIXPKGS_NVIMTREESITTER_FOLDER,
                "generated.nix"
            )
            open(os.path.join(args.nixpkgs, treesitter_generated_nix_path), "w").write(generated)

            if self.nixpkgs_repo:
                index = self.nixpkgs_repo.index
                for diff in index.diff(None):
                    if diff.a_path == treesitter_generated_nix_path:
                        msg = "vimPlugins.nvim-treesitter: update grammars"
                        print(f"committing to nixpkgs: {msg}")
                        index.add([treesitter_generated_nix_path])
                        index.commit(msg)
                        return
                print("no updates to nvim-treesitter grammars")


def main():
    global luaPlugins

    log.debug(f"Loading from {ROOT}/../get-plugins.nix")
    with open(f"{ROOT}/../get-plugins.nix") as f:
        GET_PLUGINS = f.read()
    editor = VimEditor(
        "vim", Path("pkgs/applications/editors/vim/plugins"), GET_PLUGINS
    )
    editor.run()


if __name__ == "__main__":
    main()
